# -*- coding: utf-8 -*-
"""
Local HTTP Proxy Server for BoomerStreamer
Streams video through urllib to bypass TLS fingerprint blocking
Improved version with better buffering and seeking support
"""

import threading
import socket
import urllib.request
import urllib.error
import xbmc
from http.server import HTTPServer, BaseHTTPRequestHandler
from socketserver import ThreadingMixIn
from urllib.parse import urlparse, parse_qs, quote
import time
import ssl


class ThreadingHTTPServer(ThreadingMixIn, HTTPServer):
    """HTTP server that handles each request in a new thread"""
    daemon_threads = True
    allow_reuse_address = True


class StreamProxy:
    """Manages a local HTTP proxy for streaming with improved buffering"""

    _instance = None
    _lock = threading.Lock()

    DEFAULT_HEADERS = {
        'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
        'Accept': '*/*',
        'Accept-Language': 'cs,en;q=0.9',
        'Accept-Encoding': 'identity',
    }

    def __new__(cls):
        if cls._instance is None:
            with cls._lock:
                if cls._instance is None:
                    cls._instance = super().__new__(cls)
                    cls._instance._initialized = False
        return cls._instance

    def __init__(self):
        if self._initialized:
            return
        self._initialized = True
        self.server = None
        self.server_thread = None
        self.port = None
        self.content_length_cache = {}

        # Create SSL context that doesn't verify certificates (for compatibility)
        self.ssl_context = ssl.create_default_context()
        self.ssl_context.check_hostname = False
        self.ssl_context.verify_mode = ssl.CERT_NONE

    def _make_request(self, url, headers, method='GET', timeout=30):
        """Make HTTP request using urllib"""
        req = urllib.request.Request(url, headers=headers, method=method)
        return urllib.request.urlopen(req, timeout=timeout, context=self.ssl_context)

    def _find_free_port(self):
        """Find a free port on localhost"""
        with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as s:
            s.bind(('127.0.0.1', 0))
            return s.getsockname()[1]

    def start_server(self):
        """Start the proxy server if not already running"""
        if self.server is not None:
            return self.port

        self.port = self._find_free_port()
        proxy = self

        class ProxyHandler(BaseHTTPRequestHandler):
            # Chunk size for streaming (2MB for smoother playback)
            CHUNK_SIZE = 2 * 1024 * 1024

            def log_message(self, format, *args):
                xbmc.log(f"[BoomerStreamer Proxy] {format % args}", xbmc.LOGDEBUG)

            def _get_content_length(self, stream_url, headers):
                """Get content length from cache or HEAD request"""
                if stream_url in proxy.content_length_cache:
                    cached = proxy.content_length_cache[stream_url]
                    if time.time() - cached['time'] < 3600:
                        return cached['length']

                try:
                    response = proxy._make_request(stream_url, headers, method='HEAD', timeout=10)
                    content_length = response.getheader('Content-Length')
                    response.close()
                    if content_length:
                        length = int(content_length)
                        proxy.content_length_cache[stream_url] = {
                            'length': length,
                            'time': time.time()
                        }
                        return length
                except Exception as e:
                    xbmc.log(f"[BoomerStreamer Proxy] HEAD failed: {e}", xbmc.LOGDEBUG)
                return None

            def do_GET(self):
                response = None
                try:
                    parsed = urlparse(self.path)
                    params = parse_qs(parsed.query)

                    stream_url = params.get('url', [''])[0]
                    referer = params.get('referer', ['https://hellspy.to'])[0]
                    cookies = params.get('cookies', [''])[0]

                    if not stream_url:
                        self.send_error(400, 'Missing URL parameter')
                        return

                    req_headers = {
                        'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
                        'Referer': referer,
                        'Accept': '*/*',
                        'Accept-Encoding': 'identity',
                        'Connection': 'keep-alive',
                    }

                    if 'hellspy' in referer:
                        req_headers['Origin'] = 'https://hellspy.to'

                    if cookies:
                        req_headers['Cookie'] = cookies

                    total_length = self._get_content_length(stream_url, req_headers)

                    range_header = self.headers.get('Range')
                    start_byte = 0
                    end_byte = None

                    if range_header:
                        req_headers['Range'] = range_header
                        try:
                            range_spec = range_header.replace('bytes=', '')
                            parts = range_spec.split('-')
                            start_byte = int(parts[0]) if parts[0] else 0
                            end_byte = int(parts[1]) if len(parts) > 1 and parts[1] else None
                        except:
                            pass
                        xbmc.log(f"[BoomerStreamer Proxy] Range: {start_byte}-{end_byte or 'end'} of {total_length}", xbmc.LOGINFO)

                    response = proxy._make_request(stream_url, req_headers, timeout=60)

                    resp_length = response.getheader('Content-Length')
                    resp_range = response.getheader('Content-Range')
                    content_type = response.getheader('Content-Type') or 'video/mp4'
                    status_code = response.status

                    xbmc.log(f"[BoomerStreamer Proxy] Response: {status_code}, Length: {resp_length}, Range: {resp_range}", xbmc.LOGINFO)

                    if status_code == 206 or range_header:
                        self.send_response(206)

                        if resp_range:
                            self.send_header('Content-Range', resp_range)
                        elif total_length:
                            chunk_len = int(resp_length) if resp_length else (total_length - start_byte)
                            end = start_byte + chunk_len - 1
                            self.send_header('Content-Range', f'bytes {start_byte}-{end}/{total_length}')

                        if resp_length:
                            self.send_header('Content-Length', resp_length)
                    else:
                        self.send_response(200)
                        if resp_length:
                            self.send_header('Content-Length', resp_length)
                        elif total_length:
                            self.send_header('Content-Length', str(total_length))

                    self.send_header('Content-Type', content_type)
                    self.send_header('Accept-Ranges', 'bytes')
                    self.send_header('Connection', 'keep-alive')
                    self.send_header('Cache-Control', 'no-cache')
                    self.end_headers()

                    bytes_sent = 0
                    try:
                        while True:
                            chunk = response.read(self.CHUNK_SIZE)
                            if not chunk:
                                break
                            self.wfile.write(chunk)
                            bytes_sent += len(chunk)
                    except (BrokenPipeError, ConnectionResetError, ConnectionAbortedError):
                        xbmc.log(f"[BoomerStreamer Proxy] Client disconnected after {bytes_sent} bytes", xbmc.LOGDEBUG)
                    except Exception as e:
                        xbmc.log(f"[BoomerStreamer Proxy] Stream error after {bytes_sent} bytes: {e}", xbmc.LOGERROR)
                    finally:
                        if response:
                            response.close()

                except socket.timeout:
                    xbmc.log("[BoomerStreamer Proxy] Request timeout", xbmc.LOGERROR)
                    self.send_error(504, 'Gateway Timeout')
                except urllib.error.HTTPError as e:
                    xbmc.log(f"[BoomerStreamer Proxy] HTTP error: {e.code} {e.reason}", xbmc.LOGERROR)
                    self.send_error(502, f'Upstream error: {e.code} {e.reason}')
                except urllib.error.URLError as e:
                    xbmc.log(f"[BoomerStreamer Proxy] URL error: {e}", xbmc.LOGERROR)
                    self.send_error(502, f'Upstream error: {str(e)}')
                except Exception as e:
                    xbmc.log(f"[BoomerStreamer Proxy] Error: {e}", xbmc.LOGERROR)
                    try:
                        self.send_error(500, str(e))
                    except:
                        pass

            def do_HEAD(self):
                try:
                    parsed = urlparse(self.path)
                    params = parse_qs(parsed.query)

                    stream_url = params.get('url', [''])[0]
                    referer = params.get('referer', ['https://hellspy.to'])[0]
                    cookies = params.get('cookies', [''])[0]

                    if not stream_url:
                        self.send_error(400, 'Missing URL parameter')
                        return

                    headers = {
                        'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
                        'Referer': referer,
                    }
                    if cookies:
                        headers['Cookie'] = cookies

                    response = proxy._make_request(stream_url, headers, method='HEAD', timeout=15)

                    self.send_response(200)

                    content_type = response.getheader('Content-Type') or 'video/mp4'
                    self.send_header('Content-Type', content_type)

                    content_length = response.getheader('Content-Length')
                    if content_length:
                        self.send_header('Content-Length', content_length)

                    self.send_header('Accept-Ranges', 'bytes')
                    self.end_headers()
                    response.close()

                except Exception as e:
                    xbmc.log(f"[BoomerStreamer Proxy] HEAD error: {e}", xbmc.LOGERROR)
                    self.send_error(500, str(e))

        self.server = ThreadingHTTPServer(('127.0.0.1', self.port), ProxyHandler)
        self.server.timeout = None

        def serve_forever():
            xbmc.log(f"[BoomerStreamer Proxy] Server started on port {self.port}", xbmc.LOGINFO)
            self.server.serve_forever()

        self.server_thread = threading.Thread(target=serve_forever, daemon=True)
        self.server_thread.start()

        time.sleep(0.2)

        return self.port

    def stop_server(self):
        """Stop the proxy server"""
        if self.server:
            xbmc.log("[BoomerStreamer Proxy] Stopping server", xbmc.LOGINFO)
            try:
                self.server.shutdown()
            except:
                pass
            self.server = None

    def get_proxy_url(self, stream_url, referer='https://hellspy.to', cookies=''):
        """Get the proxy URL for a stream"""
        port = self.start_server()

        encoded_url = quote(stream_url, safe='')
        encoded_referer = quote(referer, safe='')
        encoded_cookies = quote(cookies, safe='')

        proxy_url = f"http://127.0.0.1:{port}/stream?url={encoded_url}&referer={encoded_referer}&cookies={encoded_cookies}"

        xbmc.log(f"[BoomerStreamer Proxy] Created proxy URL for: {stream_url[:80]}...", xbmc.LOGINFO)

        return proxy_url


_proxy = None

def get_proxy():
    """Get the global proxy instance"""
    global _proxy
    if _proxy is None:
        _proxy = StreamProxy()
    return _proxy
