# -*- coding: utf-8 -*-
"""
TMDB API Wrapper
Handles all communication with The Movie Database API
"""

import json
import urllib.request
import urllib.parse
import urllib.error
import xbmc
import xbmcaddon

ADDON = xbmcaddon.Addon()


class TMDBApi:
    """Wrapper for The Movie Database API"""

    BASE_URL = "https://api.themoviedb.org/3"
    IMAGE_BASE_URL = "https://image.tmdb.org/t/p"

    def __init__(self):
        # Get API key from settings or use default
        self.api_key = ADDON.getSetting('tmdb_api_key')
        if not self.api_key:
            self.api_key = "c6dc22a70f6d103b33549ff91d315855"

        # Bearer token for v4 API (optional, for future use)
        self.access_token = "eyJhbGciOiJIUzI1NiJ9.eyJhdWQiOiJjNmRjMjJhNzBmNmQxMDNiMzM1NDlmZjkxZDMxNTg1NSIsIm5iZiI6MTc2NzAzOTczMC43MTYsInN1YiI6IjY5NTJlMmYyZmI2ZTYxNzljZTRhZDhhZSIsInNjb3BlcyI6WyJhcGlfcmVhZCJdLCJ2ZXJzaW9uIjoxfQ.l1Em7-sxNDQ9zvq4VrN5G8JMGVBFD3eB2w3wsmwwCX4"

        self.language = ADDON.getSetting('language') or 'cs-CZ'
        self.region = ADDON.getSetting('region') or 'CZ'
        self.headers = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'Authorization': f'Bearer {self.access_token}'
        }

    def _make_request(self, endpoint, params=None):
        """Make a request to the TMDB API"""
        if params is None:
            params = {}

        params['api_key'] = self.api_key
        params['language'] = self.language

        query_string = urllib.parse.urlencode(params)
        url = f"{self.BASE_URL}{endpoint}?{query_string}"

        try:
            req = urllib.request.Request(url, headers=self.headers)
            with urllib.request.urlopen(req, timeout=10) as response:
                data = response.read().decode('utf-8')
                return json.loads(data)
        except urllib.error.URLError as e:
            xbmc.log(f"TMDB API Error: {str(e)}", xbmc.LOGERROR)
            return None
        except Exception as e:
            xbmc.log(f"TMDB API Error: {str(e)}", xbmc.LOGERROR)
            return None
    
    # ==================== MOVIES ====================
    
    def get_popular_movies(self, page=1):
        """Get list of popular movies"""
        data = self._make_request('/movie/popular', {'page': page, 'region': self.region})
        return data.get('results', []) if data else []
    
    def get_top_rated_movies(self, page=1):
        """Get list of top rated movies"""
        data = self._make_request('/movie/top_rated', {'page': page, 'region': self.region})
        return data.get('results', []) if data else []
    
    def get_now_playing(self, page=1):
        """Get list of movies currently in theaters"""
        data = self._make_request('/movie/now_playing', {'page': page, 'region': self.region})
        return data.get('results', []) if data else []
    
    def get_upcoming_movies(self, page=1):
        """Get list of upcoming movies"""
        data = self._make_request('/movie/upcoming', {'page': page, 'region': self.region})
        return data.get('results', []) if data else []
    
    def get_movie_details(self, movie_id):
        """Get detailed information about a movie"""
        data = self._make_request(f'/movie/{movie_id}', {
            'append_to_response': 'credits,videos,external_ids,similar,recommendations'
        })
        return data
    
    def search_movies(self, query, page=1):
        """Search for movies by title"""
        data = self._make_request('/search/movie', {
            'query': query,
            'page': page,
            'include_adult': ADDON.getSettingBool('include_adult')
        })
        return data.get('results', []) if data else []
    
    def discover_movies(self, **kwargs):
        """Discover movies with various filters"""
        params = {
            'sort_by': kwargs.get('sort_by', 'popularity.desc'),
            'page': kwargs.get('page', 1),
            'include_adult': ADDON.getSettingBool('include_adult'),
            'include_video': False
        }
        
        # Add optional filters
        if 'year' in kwargs:
            params['year'] = kwargs['year']
        if 'genre' in kwargs:
            params['with_genres'] = kwargs['genre']
        if 'min_rating' in kwargs:
            params['vote_average.gte'] = kwargs['min_rating']
        
        data = self._make_request('/discover/movie', params)
        return data.get('results', []) if data else []
    
    # ==================== TV SHOWS ====================
    
    def get_popular_tvshows(self, page=1):
        """Get list of popular TV shows"""
        data = self._make_request('/tv/popular', {'page': page})
        return data.get('results', []) if data else []
    
    def get_top_rated_tvshows(self, page=1):
        """Get list of top rated TV shows"""
        data = self._make_request('/tv/top_rated', {'page': page})
        return data.get('results', []) if data else []
    
    def get_on_air(self, page=1):
        """Get list of TV shows that are currently on the air"""
        data = self._make_request('/tv/on_the_air', {'page': page})
        return data.get('results', []) if data else []
    
    def get_airing_today(self, page=1):
        """Get list of TV shows airing today"""
        data = self._make_request('/tv/airing_today', {'page': page})
        return data.get('results', []) if data else []
    
    def get_tvshow_details(self, tv_id):
        """Get detailed information about a TV show"""
        data = self._make_request(f'/tv/{tv_id}', {
            'append_to_response': 'credits,videos,external_ids,similar,recommendations'
        })
        return data
    
    def get_season_details(self, tv_id, season_number):
        """Get detailed information about a TV season"""
        data = self._make_request(f'/tv/{tv_id}/season/{season_number}')
        return data
    
    def get_season_episodes(self, tv_id, season_number):
        """Get list of episodes for a season"""
        data = self.get_season_details(tv_id, season_number)
        return data.get('episodes', []) if data else []
    
    def get_episode_details(self, tv_id, season_number, episode_number):
        """Get detailed information about an episode"""
        data = self._make_request(f'/tv/{tv_id}/season/{season_number}/episode/{episode_number}')
        return data
    
    def search_tvshows(self, query, page=1):
        """Search for TV shows by title"""
        data = self._make_request('/search/tv', {
            'query': query,
            'page': page,
            'include_adult': ADDON.getSettingBool('include_adult')
        })
        return data.get('results', []) if data else []
    
    def discover_tvshows(self, **kwargs):
        """Discover TV shows with various filters"""
        params = {
            'sort_by': kwargs.get('sort_by', 'popularity.desc'),
            'page': kwargs.get('page', 1)
        }
        
        if 'year' in kwargs:
            params['first_air_date_year'] = kwargs['year']
        if 'genre' in kwargs:
            params['with_genres'] = kwargs['genre']
        if 'min_rating' in kwargs:
            params['vote_average.gte'] = kwargs['min_rating']
        
        data = self._make_request('/discover/tv', params)
        return data.get('results', []) if data else []
    
    # ==================== TRENDING ====================
    
    def get_trending(self, media_type='all', time_window='week', page=1):
        """Get trending items (all, movie, tv, person)"""
        data = self._make_request(f'/trending/{media_type}/{time_window}', {'page': page})
        return data.get('results', []) if data else []
    
    # ==================== GENRES ====================
    
    def get_movie_genres(self):
        """Get list of movie genres"""
        data = self._make_request('/genre/movie/list')
        return data.get('genres', []) if data else []
    
    def get_tv_genres(self):
        """Get list of TV show genres"""
        data = self._make_request('/genre/tv/list')
        return data.get('genres', []) if data else []
    
    # ==================== UTILITIES ====================
    
    def get_external_ids(self, media_type, media_id):
        """Get external IDs (IMDB, TVDB, etc.) for a movie or TV show"""
        endpoint = f'/{media_type}/{media_id}/external_ids'
        return self._make_request(endpoint)
    
    def find_by_external_id(self, external_id, source='imdb_id'):
        """Find content by external ID (e.g., IMDB ID)"""
        data = self._make_request('/find/' + external_id, {'external_source': source})
        return data
    
    def get_image_url(self, path, size='w500'):
        """Get full image URL"""
        if path:
            return f"{self.IMAGE_BASE_URL}/{size}{path}"
        return None
    
    def get_configuration(self):
        """Get API configuration"""
        return self._make_request('/configuration')
