# -*- coding: utf-8 -*-
"""
Watch History and Progress Tracking for BoomerStreamer
"""

import os
import json
import time
import xbmc
import xbmcaddon
import xbmcvfs


class WatchHistory:
    """Manages watch history and progress"""
    
    def __init__(self):
        self.addon = xbmcaddon.Addon()
        self.addon_data = xbmcvfs.translatePath(self.addon.getAddonInfo('profile'))
        
        # Ensure directory exists
        if not xbmcvfs.exists(self.addon_data):
            xbmcvfs.mkdirs(self.addon_data)
        
        self.history_file = os.path.join(self.addon_data, 'watch_history.json')
        self.watched_file = os.path.join(self.addon_data, 'watched.json')
        self.progress_file = os.path.join(self.addon_data, 'progress.json')
        
        # Load data
        self.history = self._load_json(self.history_file, {'movies': [], 'tvshows': []})
        self.watched = self._load_json(self.watched_file, {'movies': [], 'episodes': []})
        self.progress = self._load_json(self.progress_file, {})
    
    def _load_json(self, filepath, default):
        """Load JSON file or return default"""
        try:
            if xbmcvfs.exists(filepath):
                with xbmcvfs.File(filepath, 'r') as f:
                    content = f.read()
                    if content:
                        return json.loads(content)
        except Exception as e:
            xbmc.log(f"[BoomerStreamer History] Load error: {e}", xbmc.LOGERROR)
        return default
    
    def _save_json(self, filepath, data):
        """Save data to JSON file"""
        try:
            with xbmcvfs.File(filepath, 'w') as f:
                f.write(json.dumps(data, indent=2, ensure_ascii=False))
            return True
        except Exception as e:
            xbmc.log(f"[BoomerStreamer History] Save error: {e}", xbmc.LOGERROR)
            return False
    
    # ==================== WATCH HISTORY ====================
    
    def add_to_history(self, media_type, media_id, title, **kwargs):
        """Add item to watch history"""
        item = {
            'media_id': media_id,
            'title': title,
            'timestamp': int(time.time()),
            **kwargs
        }
        
        # Remove if already exists
        key = 'movies' if media_type == 'movie' else 'tvshows'
        self.history[key] = [h for h in self.history[key] if h.get('media_id') != media_id]
        
        # Add to beginning
        self.history[key].insert(0, item)
        
        # Keep only last 50 items
        self.history[key] = self.history[key][:50]
        
        self._save_json(self.history_file, self.history)
        xbmc.log(f"[BoomerStreamer History] Added {media_type}: {title}", xbmc.LOGDEBUG)
    
    def get_history(self, media_type, limit=20):
        """Get watch history"""
        key = 'movies' if media_type == 'movie' else 'tvshows'
        return self.history.get(key, [])[:limit]
    
    def clear_history(self, media_type=None):
        """Clear watch history"""
        if media_type:
            key = 'movies' if media_type == 'movie' else 'tvshows'
            self.history[key] = []
        else:
            self.history = {'movies': [], 'tvshows': []}
        self._save_json(self.history_file, self.history)
    
    # ==================== WATCHED STATUS ====================
    
    def mark_watched(self, media_type, media_id, season=None, episode=None):
        """Mark item as watched"""
        if media_type == 'movie':
            if media_id not in self.watched['movies']:
                self.watched['movies'].append(media_id)
        else:
            # For episodes, store as "media_id:season:episode"
            ep_key = f"{media_id}:{season}:{episode}"
            if ep_key not in self.watched['episodes']:
                self.watched['episodes'].append(ep_key)
        
        self._save_json(self.watched_file, self.watched)
        xbmc.log(f"[BoomerStreamer History] Marked watched: {media_type} {media_id}", xbmc.LOGDEBUG)
    
    def mark_unwatched(self, media_type, media_id, season=None, episode=None):
        """Mark item as unwatched"""
        if media_type == 'movie':
            if media_id in self.watched['movies']:
                self.watched['movies'].remove(media_id)
        else:
            ep_key = f"{media_id}:{season}:{episode}"
            if ep_key in self.watched['episodes']:
                self.watched['episodes'].remove(ep_key)
        
        self._save_json(self.watched_file, self.watched)
    
    def is_watched(self, media_type, media_id, season=None, episode=None):
        """Check if item is watched"""
        if media_type == 'movie':
            return media_id in self.watched['movies']
        else:
            ep_key = f"{media_id}:{season}:{episode}"
            return ep_key in self.watched['episodes']
    
    def get_watched_episodes(self, media_id):
        """Get list of watched episodes for a TV show"""
        watched = []
        for ep_key in self.watched['episodes']:
            parts = ep_key.split(':')
            if len(parts) == 3 and parts[0] == str(media_id):
                watched.append({
                    'season': int(parts[1]),
                    'episode': int(parts[2])
                })
        return watched
    
    def get_first_unwatched_episode(self, media_id, seasons_data):
        """
        Get the first unwatched episode for a TV show
        seasons_data: list of seasons with episode counts
        Returns: (season, episode) or None if all watched
        """
        watched = self.get_watched_episodes(media_id)
        watched_set = {(w['season'], w['episode']) for w in watched}
        
        for season in sorted(seasons_data, key=lambda s: s.get('season_number', 0)):
            season_num = season.get('season_number', 0)
            if season_num == 0:  # Skip specials
                continue
            
            episode_count = season.get('episode_count', 0)
            for ep_num in range(1, episode_count + 1):
                if (season_num, ep_num) not in watched_set:
                    return (season_num, ep_num)
        
        return None  # All episodes watched
    
    def mark_all_watched(self, media_id, seasons_data):
        """Mark all episodes of a TV show as watched"""
        for season in seasons_data:
            season_num = season.get('season_number', 0)
            if season_num == 0:
                continue
            episode_count = season.get('episode_count', 0)
            for ep_num in range(1, episode_count + 1):
                self.mark_watched('tv', media_id, season_num, ep_num)
    
    def mark_all_unwatched(self, media_id):
        """Mark all episodes of a TV show as unwatched"""
        self.watched['episodes'] = [
            ep for ep in self.watched['episodes'] 
            if not ep.startswith(f"{media_id}:")
        ]
        self._save_json(self.watched_file, self.watched)
    
    # ==================== PLAYBACK PROGRESS ====================
    
    def save_progress(self, media_type, media_id, position, duration, season=None, episode=None):
        """Save playback progress"""
        if duration <= 0:
            return
        
        percent = (position / duration) * 100
        
        if media_type == 'movie':
            key = f"movie:{media_id}"
        else:
            key = f"episode:{media_id}:{season}:{episode}"
        
        self.progress[key] = {
            'position': position,
            'duration': duration,
            'percent': percent,
            'timestamp': int(time.time())
        }
        
        # Auto-mark as watched if > 90%
        if percent >= 90:
            if media_type == 'movie':
                self.mark_watched('movie', media_id)
            else:
                self.mark_watched('tv', media_id, season, episode)
            # Remove progress entry
            del self.progress[key]
        
        self._save_json(self.progress_file, self.progress)
    
    def get_progress(self, media_type, media_id, season=None, episode=None):
        """Get playback progress"""
        if media_type == 'movie':
            key = f"movie:{media_id}"
        else:
            key = f"episode:{media_id}:{season}:{episode}"
        
        return self.progress.get(key, None)
    
    def clear_progress(self, media_type, media_id, season=None, episode=None):
        """Clear playback progress"""
        if media_type == 'movie':
            key = f"movie:{media_id}"
        else:
            key = f"episode:{media_id}:{season}:{episode}"
        
        if key in self.progress:
            del self.progress[key]
            self._save_json(self.progress_file, self.progress)


# Singleton
_watch_history = None

def get_watch_history():
    global _watch_history
    if _watch_history is None:
        _watch_history = WatchHistory()
    return _watch_history
