# -*- coding: utf-8 -*-
"""
Webshare.cz API Client for BoomerStreamer
"""

import hashlib
import urllib.request
import urllib.parse
import urllib.error
import xml.etree.ElementTree as ET
import xbmc
import xbmcaddon


class WebshareAPI:
    """Webshare.cz API client"""

    API_URL = "https://webshare.cz/api"

    def __init__(self):
        self.headers = {
            'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            'Accept': 'text/xml; charset=UTF-8',
            'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8'
        }
        self.token = None
        self.logged_in = False

        # Load credentials from settings
        addon = xbmcaddon.Addon()
        self.username = addon.getSetting('webshare_username') or 'madmaxx'
        self.password = addon.getSetting('webshare_password') or 'pepazdepapepazdepa'

    def _api_call(self, endpoint, data=None):
        """Make API call and return parsed XML response"""
        url = f"{self.API_URL}/{endpoint}/"

        if data is None:
            data = {}

        # Add token if logged in
        if self.token:
            data['wst'] = self.token

        try:
            post_data = urllib.parse.urlencode(data).encode('utf-8')
            req = urllib.request.Request(url, data=post_data, headers=self.headers)
            with urllib.request.urlopen(req, timeout=15) as response:
                content = response.read()

            # Log raw response for debug
            xbmc.log(f"[BoomerStreamer Webshare] API {endpoint} response length: {len(content)}", xbmc.LOGINFO)

            # Log first 500 chars of response for debug (only for search)
            if endpoint == 'search':
                response_preview = content[:800].decode('utf-8', errors='replace')
                xbmc.log(f"[BoomerStreamer Webshare] Response: {response_preview}", xbmc.LOGINFO)

            # Parse XML response
            root = ET.fromstring(content)
            status = root.find('status')

            if status is not None and status.text == 'OK':
                return root
            else:
                code = root.find('code')
                message = root.find('message')
                error_msg = message.text if message is not None else 'Unknown error'
                error_code = code.text if code is not None else 'N/A'
                xbmc.log(f"[BoomerStreamer Webshare] API Error [{error_code}]: {error_msg}", xbmc.LOGERROR)
                return None

        except Exception as e:
            xbmc.log(f"[BoomerStreamer Webshare] Request error: {e}", xbmc.LOGERROR)
            return None
    
    def _md5_crypt(self, password, salt):
        """
        Pure Python MD5-crypt implementation
        Compatible with Unix crypt(3) $1$ format
        """
        # Ensure we're working with bytes
        if isinstance(password, str):
            password = password.encode('utf-8')
        if isinstance(salt, str):
            salt = salt.encode('utf-8')
        
        # Limit salt to 8 characters
        salt = salt[:8]
        
        # The MD5 crypt magic prefix
        magic = b'$1$'
        
        # Start with password + magic + salt
        ctx = hashlib.md5()
        ctx.update(password)
        ctx.update(magic)
        ctx.update(salt)
        
        # Then add password + salt + password (md5 of this)
        ctx1 = hashlib.md5()
        ctx1.update(password)
        ctx1.update(salt)
        ctx1.update(password)
        final = ctx1.digest()
        
        # Add alternating bits of final to ctx
        pl = len(password)
        i = pl
        while i > 0:
            if i > 16:
                ctx.update(final[:16])
            else:
                ctx.update(final[:i])
            i -= 16
        
        # Clear final
        final = b'\x00' * 16
        
        # Add bits from password based on password length
        i = len(password)
        while i:
            if i & 1:
                ctx.update(b'\x00')
            else:
                ctx.update(password[:1])
            i >>= 1
        
        final = ctx.digest()
        
        # Now do 1000 iterations
        for i in range(1000):
            ctx1 = hashlib.md5()
            
            if i & 1:
                ctx1.update(password)
            else:
                ctx1.update(final)
            
            if i % 3:
                ctx1.update(salt)
            
            if i % 7:
                ctx1.update(password)
            
            if i & 1:
                ctx1.update(final)
            else:
                ctx1.update(password)
            
            final = ctx1.digest()
        
        # Convert to the crypt(3) base64 encoding
        itoa64 = b'./0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz'
        
        def to64(v, n):
            result = b''
            while n > 0:
                result += bytes([itoa64[v & 0x3f]])
                v >>= 6
                n -= 1
            return result
        
        rearranged = b''
        rearranged += to64((final[0] << 16) | (final[6] << 8) | final[12], 4)
        rearranged += to64((final[1] << 16) | (final[7] << 8) | final[13], 4)
        rearranged += to64((final[2] << 16) | (final[8] << 8) | final[14], 4)
        rearranged += to64((final[3] << 16) | (final[9] << 8) | final[15], 4)
        rearranged += to64((final[4] << 16) | (final[10] << 8) | final[5], 4)
        rearranged += to64(final[11], 2)
        
        result = magic + salt + b'$' + rearranged
        return result.decode('utf-8')
    
    def get_salt(self):
        """Get password salt for user"""
        root = self._api_call('salt', {'username_or_email': self.username})
        if root is not None:
            salt_elem = root.find('salt')
            if salt_elem is not None:
                return salt_elem.text
        return None
    
    def login(self):
        """Login to Webshare"""
        if self.logged_in and self.token:
            return True
        
        xbmc.log(f"[BoomerStreamer Webshare] Logging in as {self.username}", xbmc.LOGINFO)
        
        # Get salt
        salt = self.get_salt()
        if not salt:
            xbmc.log("[BoomerStreamer Webshare] Failed to get salt", xbmc.LOGERROR)
            return False
        
        xbmc.log(f"[BoomerStreamer Webshare] Got salt: {salt}", xbmc.LOGDEBUG)
        
        # Create password hash: SHA1(MD5_CRYPT(password))
        try:
            md5_crypted = self._md5_crypt(self.password, salt)
            password_hash = hashlib.sha1(md5_crypted.encode('utf-8')).hexdigest()
            
            xbmc.log(f"[BoomerStreamer Webshare] Password hash created", xbmc.LOGDEBUG)
            
            # Login
            root = self._api_call('login', {
                'username_or_email': self.username,
                'password': password_hash,
                'keep_logged_in': '1'
            })
            
            if root is not None:
                token_elem = root.find('token')
                if token_elem is not None:
                    self.token = token_elem.text
                    self.logged_in = True
                    xbmc.log(f"[BoomerStreamer Webshare] Login successful, token: {self.token[:20]}...", xbmc.LOGINFO)
                    return True
        except Exception as e:
            xbmc.log(f"[BoomerStreamer Webshare] Login error: {e}", xbmc.LOGERROR)
        
        return False
    
    # Common words to ignore in relevance calculation (stop words)
    STOP_WORDS = {
        'the', 'a', 'an', 'of', 'and', 'or', 'in', 'on', 'at', 'to', 'for',
        'is', 'it', 'be', 'as', 'by', 'with', 'from', 'that', 'this',
        # Common video file terms
        'mkv', 'avi', 'mp4', 'x264', 'x265', 'hevc', 'avc', 'bluray', 'brrip',
        'webrip', 'web', 'dl', 'hdtv', 'dvdrip', 'hdrip', 'remux', 'proper',
        'repack', 'dubbed', 'dual', 'multi', 'complete', 'extended', 'unrated',
        'directors', 'cut', 'edition', 'remastered', 'imax', 'dts', 'ac3',
        'aac', 'truehd', 'atmos', 'hdr', 'sdr', '10bit', '8bit'
    }

    def _remove_diacritics(self, text):
        """Remove diacritics from text for better matching"""
        import unicodedata
        # Normalize to NFD (decomposed form), then remove combining characters
        normalized = unicodedata.normalize('NFD', text)
        # Remove combining diacritical marks
        result = ''.join(c for c in normalized if unicodedata.category(c) != 'Mn')
        return result

    def _calculate_relevance(self, query, filename):
        """
        Calculate relevance score between query and filename.
        Returns a score from 0.0 (no match) to 1.0 (perfect match).
        Requires ALL significant query words to be present in filename.
        """
        if not query or not filename:
            return 0.0

        # Normalize strings: lowercase, remove diacritics, remove special chars
        import re
        query_norm = self._remove_diacritics(query.lower())
        query_norm = re.sub(r'[^\w\s]', ' ', query_norm)
        query_norm = re.sub(r'\s+', ' ', query_norm).strip()

        filename_norm = self._remove_diacritics(filename.lower())
        filename_norm = re.sub(r'[^\w\s]', ' ', filename_norm)
        filename_norm = re.sub(r'\s+', ' ', filename_norm).strip()

        # Split into words
        query_all_words = query_norm.split()
        filename_all_words = filename_norm.split()

        # Filter out stop words and short words for relevance calculation
        query_words = set(w for w in query_all_words if w not in self.STOP_WORDS and len(w) > 1)
        filename_words = set(w for w in filename_all_words if w not in self.STOP_WORDS and len(w) > 1)

        if not query_words:
            # If query is only stop words, fall back to original behavior
            query_words = set(query_all_words)
            filename_words = set(filename_all_words)

        if not query_words:
            return 0.0

        # Check how many query words are in filename
        matching_words = query_words & filename_words

        # STRICT: ALL query words must be present for a valid match
        # If any significant word is missing, return 0
        missing_words = query_words - filename_words
        if missing_words:
            # Allow partial matches only if at least 80% of words match
            # This helps with slight variations but requires most words to be present
            match_ratio = len(matching_words) / len(query_words)
            if match_ratio < 0.8:
                return 0.0

        # Calculate base score from word overlap
        word_match_ratio = len(matching_words) / len(query_words)

        # Bonus for exact substring match (using normalized strings)
        exact_match_bonus = 0.0
        if query_norm in filename_norm:
            exact_match_bonus = 0.3

        # Bonus if filename starts with query (higher priority)
        starts_with_bonus = 0.0
        if filename_norm.startswith(query_norm):
            starts_with_bonus = 0.2

        # Calculate final score
        score = word_match_ratio + exact_match_bonus + starts_with_bonus

        # Cap at 1.0
        return min(score, 1.0)

    def search(self, query, category='video', limit=50, min_relevance=0.6):
        """
        Search for files with relevance filtering.

        Args:
            query: Search query string
            category: File category (default: 'video')
            limit: Maximum number of results to request from API
            min_relevance: Minimum relevance score (0.0-1.0) to include results.
                          Default 0.6 filters out irrelevant results.
        """
        if not self.login():
            xbmc.log("[BoomerStreamer Webshare] Not logged in, search may be limited", xbmc.LOGWARNING)

        xbmc.log(f"[BoomerStreamer Webshare] Searching: {query} (min_relevance={min_relevance})", xbmc.LOGINFO)

        root = self._api_call('search', {
            'what': query,
            'category': category,
            'limit': str(limit),
            'sort': 'largest'  # Prefer larger files (better quality)
        })

        results = []
        if root is not None:
            # Log total count if available
            total = root.find('total')
            if total is not None:
                xbmc.log(f"[BoomerStreamer Webshare] API returned total: {total.text}", xbmc.LOGINFO)

            # Debug: log all child element tags
            child_tags = [child.tag for child in root]
            xbmc.log(f"[BoomerStreamer Webshare] XML children: {child_tags[:20]}", xbmc.LOGINFO)

            for file_elem in root.findall('file'):
                try:
                    ident = file_elem.find('ident')
                    name = file_elem.find('name')  # API returns 'name' not 'n'
                    size = file_elem.find('size')
                    ftype = file_elem.find('type')

                    if ident is not None and name is not None:
                        filename = name.text

                        # Calculate relevance score
                        relevance = self._calculate_relevance(query, filename)

                        # Filter by minimum relevance
                        if relevance >= min_relevance:
                            file_info = {
                                'ident': ident.text,
                                'name': filename,
                                'size': int(size.text) if size is not None and size.text else 0,
                                'type': ftype.text if ftype is not None else '',
                                'source': 'webshare',
                                'relevance': relevance
                            }
                            results.append(file_info)
                            xbmc.log(f"[BoomerStreamer Webshare] Found (relevance={relevance:.2f}): {filename[:60] if filename else 'N/A'}...", xbmc.LOGINFO)
                        else:
                            xbmc.log(f"[BoomerStreamer Webshare] Filtered out (relevance={relevance:.2f}): {filename[:60] if filename else 'N/A'}...", xbmc.LOGDEBUG)
                except Exception as e:
                    xbmc.log(f"[BoomerStreamer Webshare] Parse error: {e}", xbmc.LOGERROR)
        else:
            xbmc.log("[BoomerStreamer Webshare] Search returned None/Error", xbmc.LOGWARNING)

        # Sort by relevance (highest first), then by size
        results.sort(key=lambda x: (x.get('relevance', 0), x.get('size', 0)), reverse=True)

        xbmc.log(f"[BoomerStreamer Webshare] Found {len(results)} relevant results (filtered from API response)", xbmc.LOGINFO)
        return results
    
    def get_file_link(self, ident):
        """Get direct download link for file"""
        if not self.login():
            xbmc.log("[BoomerStreamer Webshare] Must be logged in to get file link", xbmc.LOGERROR)
            return None
        
        root = self._api_call('file_link', {
            'ident': ident,
            'download_type': 'video_stream',
            'force_https': '1'
        })
        
        if root is not None:
            link_elem = root.find('link')
            if link_elem is not None:
                xbmc.log(f"[BoomerStreamer Webshare] Got link: {link_elem.text[:80]}...", xbmc.LOGINFO)
                return link_elem.text
        
        return None
    
    def get_file_info(self, ident):
        """Get file information"""
        root = self._api_call('file_info', {'ident': ident})
        
        if root is not None:
            return {
                'name': root.find('n').text if root.find('n') is not None else '',
                'size': int(root.find('size').text) if root.find('size') is not None else 0,
                'type': root.find('type').text if root.find('type') is not None else '',
            }
        return None


# Singleton instance
_webshare = None

def get_webshare():
    global _webshare
    if _webshare is None:
        _webshare = WebshareAPI()
    return _webshare
