# -*- coding: utf-8 -*-
"""
TMDB Movie Scraper for Kodi
Main entry point for the addon
"""

import sys
import os
import urllib.parse

import xbmc
import xbmcgui
import xbmcplugin
import xbmcaddon

# Get addon info
ADDON = xbmcaddon.Addon()
ADDON_PATH = ADDON.getAddonInfo('path')
ADDON_ID = ADDON.getAddonInfo('id')
ADDON_NAME = ADDON.getAddonInfo('name')

from datetime import datetime, timedelta

# Add lib folder to path
LIB_PATH = os.path.join(ADDON_PATH, 'lib')
if LIB_PATH not in sys.path:
    sys.path.insert(0, LIB_PATH)

# Import our custom modules
from tmdb import TMDBApi
from scraper import StreamScraper, ResolverManager
from proxy import get_proxy
from webshare import get_webshare
from watch_history import get_watch_history

# Get plugin handle and base URL
HANDLE = int(sys.argv[1]) if len(sys.argv) > 1 else -1
BASE_URL = sys.argv[0] if len(sys.argv) > 0 else ''

# Initialize APIs
tmdb = TMDBApi()
scraper = StreamScraper()
resolver = ResolverManager()
webshare = get_webshare()
watch_history = get_watch_history()


def is_asian_content(item):
    """Check if content is from Asian countries with non-Latin scripts (Korean, Chinese, Japanese, Thai, etc.)"""
    origin_countries = item.get('origin_country', [])
    original_language = item.get('original_language', '')

    # Asian countries with non-Latin scripts
    asian_countries = {'KR', 'CN', 'TW', 'HK', 'JP', 'TH', 'IN', 'PH'}
    # Asian languages with non-Latin scripts
    asian_languages = {'ko', 'zh', 'ja', 'th', 'hi', 'ta', 'te', 'ml', 'kn', 'bn', 'tl'}

    for country in origin_countries:
        if country in asian_countries:
            return True
    if original_language in asian_languages:
        return True
    return False


def is_content_too_old(item, media_type='movie'):
    """Check if content is older than the configured max age"""
    try:
        max_age_months = int(ADDON.getSetting('max_age_months') or '0')
    except (ValueError, TypeError):
        max_age_months = 0

    if max_age_months == 0:
        return False  # Filtering disabled

    cutoff_date = datetime.now() - timedelta(days=max_age_months * 30)

    if media_type == 'movie':
        release_date = item.get('release_date', '')
    else:
        release_date = item.get('first_air_date', '')

    if not release_date:
        return False  # Keep items without date

    try:
        item_date = datetime.strptime(release_date, '%Y-%m-%d')
        return item_date < cutoff_date
    except (ValueError, TypeError):
        return False


def filter_content(items, media_type='movie', filter_asian=True, filter_old=True):
    """Filter content based on settings"""
    filtered = []
    for item in items:
        # Filter Asian content with non-Latin scripts
        if filter_asian and is_asian_content(item):
            continue

        # Filter old content
        if filter_old and is_content_too_old(item, media_type):
            continue

        filtered.append(item)

    return filtered


def get_url(**kwargs):
    """Create a URL for calling the plugin recursively"""
    return '{}?{}'.format(BASE_URL, urllib.parse.urlencode(kwargs))


def get_params():
    """Parse URL parameters"""
    params = {}
    paramstring = sys.argv[2]
    if paramstring:
        params = dict(urllib.parse.parse_qsl(paramstring[1:]))
    return params


def show_main_menu():
    """Show the main menu of the addon"""
    xbmcplugin.setPluginCategory(HANDLE, ADDON_NAME)
    xbmcplugin.setContent(HANDLE, 'files')
    
    menu_items = [
        {'title': '>> Pokracovat v sledovani', 'action': 'continue_watching', 'icon': 'DefaultRecentlyAddedMovies.png'},
        {'title': '[F] Posledni filmy', 'action': 'recent_movies', 'icon': 'DefaultRecentlyAddedMovies.png'},
        {'title': '[S] Posledni serialy', 'action': 'recent_tvshows', 'icon': 'DefaultRecentlyAddedEpisodes.png'},
        {'title': '[*] Popularni filmy', 'action': 'popular_movies', 'icon': 'DefaultMovies.png'},
        {'title': '[*] Popularni serialy', 'action': 'popular_tvshows', 'icon': 'DefaultTVShows.png'},
        {'title': '[+] Nejlepe hodnocene filmy', 'action': 'top_rated_movies', 'icon': 'DefaultMovies.png'},
        {'title': '[+] Nejlepe hodnocene serialy', 'action': 'top_rated_tvshows', 'icon': 'DefaultTVShows.png'},
        {'title': '[K] Prave v kinech', 'action': 'now_playing', 'icon': 'DefaultMovies.png'},
        {'title': '[V] Prave vysilane serialy', 'action': 'on_air', 'icon': 'DefaultTVShows.png'},
        {'title': '[?] Hledat filmy', 'action': 'search_movies', 'icon': 'DefaultAddonsSearch.png'},
        {'title': '[?] Hledat serialy', 'action': 'search_tvshows', 'icon': 'DefaultAddonsSearch.png'},
        {'title': '[=] Nastaveni', 'action': 'settings', 'icon': 'DefaultAddonProgram.png'},
    ]
    
    for item in menu_items:
        list_item = xbmcgui.ListItem(label=item['title'])
        list_item.setArt({'icon': item['icon'], 'thumb': item['icon']})
        url = get_url(action=item['action'])
        xbmcplugin.addDirectoryItem(HANDLE, url, list_item, isFolder=True)
    
    xbmcplugin.endOfDirectory(HANDLE)


def show_movies(movies, media_type='movie'):
    """Display a list of movies or TV shows"""
    xbmcplugin.setPluginCategory(HANDLE, ADDON_NAME)
    xbmcplugin.setContent(HANDLE, 'movies' if media_type == 'movie' else 'tvshows')
    
    for movie in movies:
        # Get movie/show info
        title = movie.get('title') or movie.get('name', 'Unknown')
        movie_id = movie.get('id')
        overview = movie.get('overview', '')
        rating = movie.get('vote_average', 0)
        vote_count = movie.get('vote_count', 0)
        year = ''
        
        release_date = movie.get('release_date') or movie.get('first_air_date', '')
        if release_date:
            year = release_date[:4]
        
        # Build poster and fanart URLs
        poster_path = movie.get('poster_path', '')
        backdrop_path = movie.get('backdrop_path', '')
        poster_url = f"https://image.tmdb.org/t/p/w500{poster_path}" if poster_path else ''
        fanart_url = f"https://image.tmdb.org/t/p/original{backdrop_path}" if backdrop_path else ''
        
        # Create list item - format: Title (Year) ★ 7.5
        label = title
        if year:
            label += f" ({year})"
        if rating > 0:
            label += f"  ★ {rating:.1f}"
        
        list_item = xbmcgui.ListItem(label=label)
        
        # Set artwork
        list_item.setArt({
            'poster': poster_url,
            'thumb': poster_url,
            'fanart': fanart_url,
            'icon': poster_url
        })
        
        # Set info labels - use full label with year/rating for title too
        info_tag = list_item.getVideoInfoTag()
        info_tag.setTitle(label)  # Use label with year and rating
        info_tag.setPlot(overview)
        info_tag.setRating(rating)
        info_tag.setVotes(vote_count)
        if year:
            info_tag.setYear(int(year))
            info_tag.setPremiered(release_date)
        info_tag.setMediaType('movie' if media_type == 'movie' else 'tvshow')
        
        # Set playable properties
        list_item.setProperty('IsPlayable', 'false')
        
        # Create URL for details/play
        url = get_url(
            action='show_details',
            media_type=media_type,
            media_id=movie_id,
            title=title
        )
        
        # Add context menu
        context_menu = [
            ('Přehrát', f'RunPlugin({get_url(action="play", media_type=media_type, media_id=movie_id, title=title)})'),
            ('Vyhledat zdroje', f'RunPlugin({get_url(action="search_sources", media_type=media_type, media_id=movie_id, title=title)})'),
        ]
        list_item.addContextMenuItems(context_menu)
        
        xbmcplugin.addDirectoryItem(HANDLE, url, list_item, isFolder=True)
    
    # Add sort methods - user can change sorting in Kodi
    xbmcplugin.addSortMethod(HANDLE, xbmcplugin.SORT_METHOD_UNSORTED)  # Default order from TMDB
    xbmcplugin.addSortMethod(HANDLE, xbmcplugin.SORT_METHOD_TITLE)
    xbmcplugin.addSortMethod(HANDLE, xbmcplugin.SORT_METHOD_VIDEO_RATING)
    xbmcplugin.addSortMethod(HANDLE, xbmcplugin.SORT_METHOD_VIDEO_YEAR)
    xbmcplugin.addSortMethod(HANDLE, xbmcplugin.SORT_METHOD_DATE)
    xbmcplugin.endOfDirectory(HANDLE)


def show_details(media_type, media_id, title):
    """Show details for a movie or TV show - for movies, search and show sources directly"""
    xbmcplugin.setPluginCategory(HANDLE, title)
    
    if media_type == 'movie':
        # For movies, directly search and show available sources
        show_sources(media_type, media_id, title)
    else:
        # For TV shows, show seasons
        xbmcplugin.setContent(HANDLE, 'seasons')
        details = tmdb.get_tvshow_details(media_id)
        
        if not details:
            xbmcgui.Dialog().notification(ADDON_NAME, 'Nepodařilo se načíst detaily', xbmcgui.NOTIFICATION_ERROR)
            return
        
        seasons = details.get('seasons', [])
        for season in seasons:
            season_number = season.get('season_number', 0)
            season_name = season.get('name', f'Série {season_number}')
            episode_count = season.get('episode_count', 0)
            poster_path = season.get('poster_path', '')
            
            label = f"{season_name} ({episode_count} epizod)"
            
            list_item = xbmcgui.ListItem(label=label)
            if poster_path:
                poster_url = f"https://image.tmdb.org/t/p/w500{poster_path}"
                list_item.setArt({'poster': poster_url, 'thumb': poster_url})
            
            url = get_url(
                action='show_episodes',
                media_id=media_id,
                season=season_number,
                title=title
            )
            xbmcplugin.addDirectoryItem(HANDLE, url, list_item, isFolder=True)
        
        xbmcplugin.endOfDirectory(HANDLE)


def show_sources(media_type, media_id, title, season=None, episode=None):
    """Search and display available sources from all enabled providers"""
    xbmcplugin.setContent(HANDLE, 'files')
    
    # Get enabled sources from settings
    enable_webshare = ADDON.getSetting('enable_webshare') == 'true'
    enable_hellspy = ADDON.getSetting('enable_hellspy') == 'true'
    
    # Show progress dialog
    progress = xbmcgui.DialogProgress()
    progress.create(ADDON_NAME, f'Hledám zdroje pro: {title}')
    progress.update(10, 'Načítám informace...')
    
    # Get details for year and IMDB ID
    if media_type == 'movie':
        details = tmdb.get_movie_details(media_id)
        year = details.get('release_date', '')[:4] if details and details.get('release_date') else ''
    else:
        details = tmdb.get_tvshow_details(media_id)
        year = details.get('first_air_date', '')[:4] if details and details.get('first_air_date') else ''
    
    imdb_id = details.get('imdb_id') if details else None
    
    all_streams = []
    
    # Search Hellspy
    if enable_hellspy:
        progress.update(30, 'Prohledávám Hellspy.to...')
        hellspy_streams = scraper.search_streams(
            title=title,
            year=year,
            imdb_id=imdb_id,
            media_type=media_type,
            season=season,
            episode=episode
        )
        for stream in hellspy_streams:
            stream['source'] = 'Hellspy'
        all_streams.extend(hellspy_streams)
        xbmc.log(f'[BoomerStreamer] Hellspy: {len(hellspy_streams)} streams', xbmc.LOGINFO)
    
    # Search Webshare
    if enable_webshare:
        progress.update(60, 'Prohledávám Webshare.cz...')
        
        # Clean title for search - remove special characters
        import re
        clean_title = re.sub(r'[:\-–—\(\)\[\]\'\"\.!?,]', ' ', title)
        clean_title = re.sub(r'\s+', ' ', clean_title).strip()
        
        xbmc.log(f'[BoomerStreamer] Original title: {title}', xbmc.LOGINFO)
        xbmc.log(f'[BoomerStreamer] Cleaned title: {clean_title}', xbmc.LOGINFO)
        
        # Try multiple search queries
        search_queries = []
        
        # Query 1: Clean title + year
        if year:
            search_queries.append(f'{clean_title} {year}')
        
        # Query 2: Just clean title
        search_queries.append(clean_title)
        
        # Query 3: First two words + year (for long titles)
        words = clean_title.split()
        if len(words) > 2 and year:
            search_queries.append(f'{words[0]} {words[1]} {year}')
        
        # Query 4: Just first word + year (very short search)
        if words and year:
            search_queries.append(f'{words[0]} {year}')
        
        # For TV shows, add episode format
        if media_type == 'tv' and season and episode:
            ep_query = f'{clean_title} S{int(season):02d}E{int(episode):02d}'
            search_queries.insert(0, ep_query)
        
        webshare_results = []
        seen_idents = set()
        
        for query in search_queries:
            if len(webshare_results) >= 20:
                break  # Enough results

            xbmc.log(f'[BoomerStreamer Webshare] Trying query: {query}', xbmc.LOGINFO)
            # Request more results from API since we filter by relevance
            # min_relevance=0.6 filters out unrelated results (e.g., "sedma mile" when searching "avatar")
            results = webshare.search(query, category='video', limit=100, min_relevance=0.6)
            
            for result in results:
                ident = result.get('ident', '')
                if ident and ident not in seen_idents:
                    seen_idents.add(ident)
                    webshare_results.append(result)
        
        for result in webshare_results:
            # Convert webshare result to stream format
            filename = result.get('name', '')
            size_bytes = result.get('size', 0)
            
            # Parse quality from filename
            quality = 'Unknown'
            filename_upper = filename.upper()
            if '2160P' in filename_upper or '4K' in filename_upper:
                quality = '4K'
            elif '1080P' in filename_upper or 'FULLHD' in filename_upper:
                quality = '1080p'
            elif '720P' in filename_upper or 'HD' in filename_upper:
                quality = '720p'
            elif '480P' in filename_upper:
                quality = '480p'
            
            # Parse language
            language = ''
            if 'CZ' in filename_upper or 'CZECH' in filename_upper:
                language = 'CZ'
            elif 'SK' in filename_upper or 'SLOVAK' in filename_upper:
                language = 'SK'
            elif 'EN' in filename_upper or 'ENG' in filename_upper:
                language = 'EN'
            
            # Format size
            if size_bytes > 0:
                if size_bytes >= 1024 * 1024 * 1024:
                    size = f"{size_bytes / (1024 * 1024 * 1024):.1f} GB"
                else:
                    size = f"{size_bytes / (1024 * 1024):.0f} MB"
            else:
                size = ''
            
            all_streams.append({
                'filename': filename,
                'quality': quality,
                'size': size,
                'size_bytes': size_bytes,
                'language': language,
                'source': 'Webshare',
                'ident': result.get('ident', ''),
                'url': '',  # Will be resolved on play
            })
        
        xbmc.log(f'[BoomerStreamer] Webshare: {len(webshare_results)} streams', xbmc.LOGINFO)
    
    progress.close()
    
    if not all_streams:
        xbmcgui.Dialog().notification(ADDON_NAME, 'Nebyly nalezeny žádné zdroje', xbmcgui.NOTIFICATION_WARNING)
        xbmcplugin.endOfDirectory(HANDLE, succeeded=False)
        return
    
    # Sort streams by file size (largest first)
    def get_size_bytes(stream):
        if 'size_bytes' in stream and stream['size_bytes']:
            return stream['size_bytes']
        size_str = stream.get('size', '')
        if not size_str:
            return 0
        size_str = size_str.upper().strip()
        try:
            if 'GB' in size_str:
                return float(size_str.replace('GB', '').strip()) * 1024 * 1024 * 1024
            elif 'MB' in size_str:
                return float(size_str.replace('MB', '').strip()) * 1024 * 1024
        except:
            pass
        return 0
    
    all_streams.sort(key=get_size_bytes, reverse=True)
    
    xbmc.log(f'[BoomerStreamer] Total: {len(all_streams)} streams', xbmc.LOGINFO)
    
    # Add to watch history
    watch_history.add_to_history(
        media_type, 
        media_id, 
        title, 
        year=year,
        season=season,
        episode=episode,
        poster=details.get('poster_path', '') if details else ''
    )
    
    # Display each stream as a list item
    for i, stream in enumerate(all_streams):
        filename = stream.get('filename', '')
        quality = stream.get('quality', 'Unknown')
        size = stream.get('size', '')
        language = stream.get('language', '')
        source = stream.get('source', 'Unknown')
        
        # Build display label with source indicator
        # Format: [WS] [1080p | CZ | 4.2 GB] Název souboru
        # or: [HS] [1080p | CZ | 4.2 GB] Název souboru
        source_tag = '[COLOR lime][WS][/COLOR]' if source == 'Webshare' else '[COLOR orange][HS][/COLOR]'
        
        meta_parts = []
        if quality and quality != 'Unknown':
            meta_parts.append(quality)
        if language:
            meta_parts.append(language)
        if size:
            meta_parts.append(size)
        
        if meta_parts:
            label = f"{source_tag} [{' | '.join(meta_parts)}] {filename if filename else f'Zdroj {i+1}'}"
        else:
            label = f"{source_tag} {filename if filename else f'Zdroj {i+1}'}"
        
        list_item = xbmcgui.ListItem(label=label)
        list_item.setProperty('IsPlayable', 'true')
        
        if size:
            list_item.setLabel2(size)
        
        info_tag = list_item.getVideoInfoTag()
        info_tag.setTitle(label)
        info_tag.setMediaType('video')
        
        # Note: setSize is not available in Kodi 21, size is shown in label instead
        
        # Create play URL based on source
        if source == 'Webshare':
            url = get_url(
                action='play_webshare',
                ident=stream.get('ident', ''),
                title=filename if filename else title,
                media_type=media_type,
                media_id=str(media_id),
                season=str(season) if season else '',
                episode=str(episode) if episode else ''
            )
        else:
            stream_url = stream.get('url', '')
            page_url = stream.get('page_url', 'https://hellspy.to')
            cookies = stream.get('cookies', '')
            url = get_url(
                action='play_stream',
                stream_url=stream_url,
                page_url=page_url,
                cookies=cookies,
                title=filename if filename else title,
                media_type=media_type,
                media_id=str(media_id),
                season=str(season) if season else '',
                episode=str(episode) if episode else ''
            )
        
        # Context menu
        context_menu = []
        if source == 'Hellspy':
            page_url = stream.get('page_url', 'https://hellspy.to')
            context_menu.append(('Otevřít v prohlížeči', f'RunPlugin({get_url(action="open_browser", url=page_url)})'))
        
        if context_menu:
            list_item.addContextMenuItems(context_menu)
        
        xbmcplugin.addDirectoryItem(HANDLE, url, list_item, isFolder=False)
    
    xbmcplugin.addSortMethod(HANDLE, xbmcplugin.SORT_METHOD_UNSORTED)
    xbmcplugin.addSortMethod(HANDLE, xbmcplugin.SORT_METHOD_SIZE)
    xbmcplugin.addSortMethod(HANDLE, xbmcplugin.SORT_METHOD_LABEL)
    xbmcplugin.endOfDirectory(HANDLE)


def show_episodes(media_id, season, title):
    """Show episodes for a TV show season"""
    xbmcplugin.setPluginCategory(HANDLE, f"{title} - Série {season}")
    xbmcplugin.setContent(HANDLE, 'episodes')
    
    episodes = tmdb.get_season_episodes(media_id, season)
    
    for episode in episodes:
        ep_number = episode.get('episode_number', 0)
        ep_name = episode.get('name', f'Epizoda {ep_number}')
        ep_overview = episode.get('overview', '')
        still_path = episode.get('still_path', '')
        air_date = episode.get('air_date', '')
        
        label = f"E{ep_number:02d} - {ep_name}"
        
        list_item = xbmcgui.ListItem(label=label)
        
        if still_path:
            still_url = f"https://image.tmdb.org/t/p/w500{still_path}"
            list_item.setArt({'thumb': still_url, 'icon': still_url})
        
        info_tag = list_item.getVideoInfoTag()
        info_tag.setTitle(ep_name)
        info_tag.setPlot(ep_overview)
        info_tag.setEpisode(ep_number)
        info_tag.setSeason(season)
        info_tag.setMediaType('episode')
        
        # Episode links to source search (isFolder=True)
        url = get_url(
            action='show_episode_sources',
            media_id=media_id,
            season=season,
            episode=ep_number,
            title=f"{title} S{season:02d}E{ep_number:02d}"
        )
        
        xbmcplugin.addDirectoryItem(HANDLE, url, list_item, isFolder=True)
    
    xbmcplugin.endOfDirectory(HANDLE)


def search_content(media_type):
    """Search for movies or TV shows"""
    keyboard = xbmc.Keyboard('', 'Hledat ' + ('filmy' if media_type == 'movie' else 'seriály'))
    keyboard.doModal()
    
    if keyboard.isConfirmed():
        query = keyboard.getText()
        if query:
            if media_type == 'movie':
                results = tmdb.search_movies(query)
            else:
                results = tmdb.search_tvshows(query)
            
            if results:
                show_movies(results, media_type)
            else:
                xbmcgui.Dialog().notification(ADDON_NAME, 'Nic nenalezeno', xbmcgui.NOTIFICATION_INFO)


def play_media(media_type, media_id, title, season=None, episode=None):
    """Play a movie or TV show episode"""
    progress = xbmcgui.DialogProgress()
    progress.create(ADDON_NAME, f'Hledám zdroje pro: {title}')
    
    # Search for streams
    progress.update(25, 'Prohledávám dostupné zdroje...')
    
    # Get IMDB ID for better search
    if media_type == 'movie':
        details = tmdb.get_movie_details(media_id)
    elif media_type == 'episode':
        details = tmdb.get_tvshow_details(media_id)
    else:
        details = tmdb.get_tvshow_details(media_id)
    
    imdb_id = details.get('imdb_id') if details else None
    
    # Search for streams using the scraper
    progress.update(50, 'Vyhledávám streamy...')
    
    streams = scraper.search_streams(
        title=title,
        year=details.get('release_date', '')[:4] if details and details.get('release_date') else '',
        imdb_id=imdb_id,
        media_type=media_type,
        season=season,
        episode=episode
    )
    
    progress.close()
    
    if not streams:
        # Show error and offer to search manually
        choice = xbmcgui.Dialog().yesno(
            ADDON_NAME,
            'Nepodařilo se najít žádné zdroje.',
            yeslabel='Hledat znovu',
            nolabel='Zrušit'
        )
        if choice:
            search_sources(media_type, media_id, title, season, episode)
        return
    
    # Let user select stream
    stream_names = []
    for s in streams:
        quality = s.get('quality', 'Unknown')
        source = s.get('source', 'Unknown')
        size = s.get('size', '')
        name = f"[{quality}] {source}"
        if size:
            name += f" ({size})"
        stream_names.append(name)
    
    selected = xbmcgui.Dialog().select('Vyberte zdroj', stream_names)
    
    if selected >= 0:
        stream = streams[selected]
        stream_url = stream.get('url', '')
        
        # Resolve the stream URL
        resolved_url = resolver.resolve(stream_url)
        
        if resolved_url:
            # Play the stream
            play_item = xbmcgui.ListItem(path=resolved_url)
            xbmcplugin.setResolvedUrl(HANDLE, True, play_item)
        else:
            xbmcgui.Dialog().notification(ADDON_NAME, 'Nepodařilo se přehrát zdroj', xbmcgui.NOTIFICATION_ERROR)


def search_sources(media_type, media_id, title, season=None, episode=None):
    """Manually search for sources"""
    keyboard = xbmc.Keyboard(title, 'Upravit hledaný výraz')
    keyboard.doModal()
    
    if keyboard.isConfirmed():
        query = keyboard.getText()
        if query:
            progress = xbmcgui.DialogProgress()
            progress.create(ADDON_NAME, f'Hledám: {query}')
            
            streams = scraper.search_streams(
                title=query,
                media_type=media_type,
                season=season,
                episode=episode
            )
            
            progress.close()
            
            if streams:
                stream_names = []
                for s in streams:
                    quality = s.get('quality', 'Unknown')
                    source = s.get('source', 'Unknown')
                    name = f"[{quality}] {source}"
                    stream_names.append(name)
                
                selected = xbmcgui.Dialog().select('Nalezené zdroje', stream_names)
                if selected >= 0:
                    stream = streams[selected]
                    resolved_url = resolver.resolve(stream.get('url', ''))
                    if resolved_url:
                        play_item = xbmcgui.ListItem(path=resolved_url)
                        xbmc.Player().play(resolved_url, play_item)
            else:
                xbmcgui.Dialog().notification(ADDON_NAME, 'Nic nenalezeno', xbmcgui.NOTIFICATION_INFO)


def router(paramstring):
    """Route to the appropriate function based on parameters"""
    params = get_params()
    action = params.get('action', '')
    
    if not action:
        show_main_menu()
    
    elif action == 'popular_movies':
        movies = tmdb.get_popular_movies()
        movies = filter_content(movies, 'movie')
        show_movies(movies, 'movie')

    elif action == 'popular_tvshows':
        shows = tmdb.get_popular_tvshows()
        shows = filter_content(shows, 'tvshow')
        show_movies(shows, 'tvshow')
    
    elif action == 'top_rated_movies':
        movies = tmdb.get_top_rated_movies()
        show_movies(movies, 'movie')
    
    elif action == 'top_rated_tvshows':
        shows = tmdb.get_top_rated_tvshows()
        show_movies(shows, 'tvshow')
    
    elif action == 'now_playing':
        movies = tmdb.get_now_playing()
        show_movies(movies, 'movie')
    
    elif action == 'on_air':
        shows = tmdb.get_on_air()
        show_movies(shows, 'tvshow')
    
    elif action == 'search_movies':
        search_content('movie')
    
    elif action == 'search_tvshows':
        search_content('tvshow')
    
    elif action == 'show_details':
        show_details(
            params.get('media_type'),
            params.get('media_id'),
            params.get('title')
        )
    
    elif action == 'show_episodes':
        show_episodes(
            params.get('media_id'),
            int(params.get('season', 1)),
            params.get('title')
        )
    
    elif action == 'show_episode_sources':
        # Show sources for a specific episode
        show_sources(
            media_type='episode',
            media_id=params.get('media_id'),
            title=params.get('title'),
            season=int(params.get('season', 1)),
            episode=int(params.get('episode', 1))
        )
    
    elif action == 'play':
        play_media(
            params.get('media_type'),
            params.get('media_id'),
            params.get('title'),
            params.get('season'),
            params.get('episode')
        )
    
    elif action == 'search_sources':
        search_sources(
            params.get('media_type'),
            params.get('media_id'),
            params.get('title'),
            params.get('season'),
            params.get('episode')
        )
    
    elif action == 'settings':
        ADDON.openSettings()
    
    elif action == 'play_stream':
        # Play stream through local proxy to bypass TLS fingerprint blocking
        stream_url = params.get('stream_url', '')
        stream_title = params.get('title', 'Video')
        page_url = params.get('page_url', 'https://hellspy.to')
        cookies = params.get('cookies', '')
        
        if stream_url:
            # Clean URL from unicode escapes
            try:
                stream_url = stream_url.encode().decode('unicode_escape')
            except:
                pass
            stream_url = stream_url.replace('\\u0026', '&')
            stream_url = stream_url.replace('&amp;', '&')
            stream_url = stream_url.replace('\\', '')
            
            xbmc.log(f'[BoomerStreamer] Original URL: {stream_url[:150]}...', xbmc.LOGINFO)
            
            # Resolve the stream URL
            resolved_url = resolver.resolve(stream_url)
            
            if resolved_url:
                # Clean resolved URL
                try:
                    resolved_url = resolved_url.encode().decode('unicode_escape')
                except:
                    pass
                resolved_url = resolved_url.replace('\\u0026', '&')
                resolved_url = resolved_url.replace('&amp;', '&')
                resolved_url = resolved_url.replace('\\', '')
                
                xbmc.log(f'[BoomerStreamer] Resolved URL: {resolved_url[:150]}...', xbmc.LOGINFO)
                
                # Get proxy URL - this streams through requests to bypass TLS blocking
                proxy = get_proxy()
                proxy_url = proxy.get_proxy_url(resolved_url, page_url, cookies)
                
                xbmc.log(f'[BoomerStreamer] Playing via proxy: {proxy_url[:100]}...', xbmc.LOGINFO)
                
                # Play through proxy
                play_item = xbmcgui.ListItem(path=proxy_url)
                play_item.setProperty('IsPlayable', 'true')
                
                # Set video info
                info_tag = play_item.getVideoInfoTag()
                info_tag.setTitle(stream_title)
                info_tag.setMediaType('video')
                
                xbmcplugin.setResolvedUrl(HANDLE, True, play_item)
            else:
                xbmcgui.Dialog().notification(ADDON_NAME, 'Nepodařilo se přehrát', xbmcgui.NOTIFICATION_ERROR)
                xbmcplugin.setResolvedUrl(HANDLE, False, xbmcgui.ListItem())
        else:
            xbmcgui.Dialog().notification(ADDON_NAME, 'Chybí URL streamu', xbmcgui.NOTIFICATION_ERROR)
            xbmcplugin.setResolvedUrl(HANDLE, False, xbmcgui.ListItem())
    
    elif action == 'open_browser':
        # Open URL in default browser
        url = params.get('url', '')
        if url:
            import webbrowser
            xbmc.log(f'[BoomerStreamer] Opening in browser: {url}', xbmc.LOGINFO)
            webbrowser.open(url)
            xbmcgui.Dialog().notification(ADDON_NAME, 'Otevřeno v prohlížeči', xbmcgui.NOTIFICATION_INFO)
    
    elif action == 'recent_movies':
        # Show recently watched movies
        show_recent_history('movie')
    
    elif action == 'recent_tvshows':
        # Show recently watched TV shows
        show_recent_history('tv')
    
    elif action == 'continue_watching':
        # Show items with progress
        show_continue_watching()
    
    elif action == 'play_webshare':
        # Play stream from Webshare
        ident = params.get('ident', '')
        stream_title = params.get('title', 'Video')
        media_type = params.get('media_type', 'movie')
        media_id = params.get('media_id', '')
        season = params.get('season', '')
        episode = params.get('episode', '')
        
        if ident:
            xbmc.log(f'[BoomerStreamer] Getting Webshare link for: {ident}', xbmc.LOGINFO)
            
            # Get direct link from Webshare
            stream_url = webshare.get_file_link(ident)
            
            if stream_url:
                xbmc.log(f'[BoomerStreamer] Webshare URL: {stream_url[:100]}...', xbmc.LOGINFO)

                # Webshare URLs need to go through proxy to bypass TLS fingerprint blocking
                proxy = get_proxy()
                proxy_url = proxy.get_proxy_url(stream_url, referer='https://webshare.cz', cookies='')

                xbmc.log(f'[BoomerStreamer] Playing Webshare via proxy: {proxy_url[:100]}...', xbmc.LOGINFO)

                play_item = xbmcgui.ListItem(path=proxy_url)
                play_item.setProperty('IsPlayable', 'true')

                info_tag = play_item.getVideoInfoTag()
                info_tag.setTitle(stream_title)
                info_tag.setMediaType('video')

                xbmcplugin.setResolvedUrl(HANDLE, True, play_item)
                
                # Mark as watched after starting playback
                if media_id:
                    if media_type == 'movie':
                        watch_history.mark_watched('movie', int(media_id))
                    elif season and episode:
                        watch_history.mark_watched('tv', int(media_id), int(season), int(episode))
            else:
                xbmcgui.Dialog().notification(ADDON_NAME, 'Nepodařilo se získat odkaz z Webshare', xbmcgui.NOTIFICATION_ERROR)
                xbmcplugin.setResolvedUrl(HANDLE, False, xbmcgui.ListItem())
        else:
            xbmcgui.Dialog().notification(ADDON_NAME, 'Chybí ID souboru', xbmcgui.NOTIFICATION_ERROR)
            xbmcplugin.setResolvedUrl(HANDLE, False, xbmcgui.ListItem())
    
    elif action == 'mark_watched':
        # Mark item as watched
        media_type = params.get('media_type', 'movie')
        media_id = params.get('media_id', '')
        season = params.get('season', '')
        episode = params.get('episode', '')
        
        if media_id:
            if media_type == 'movie':
                watch_history.mark_watched('movie', int(media_id))
                xbmcgui.Dialog().notification(ADDON_NAME, 'Označeno jako zhlédnuté', xbmcgui.NOTIFICATION_INFO)
            elif season and episode:
                watch_history.mark_watched('tv', int(media_id), int(season), int(episode))
                xbmcgui.Dialog().notification(ADDON_NAME, 'Epizoda označena jako zhlédnutá', xbmcgui.NOTIFICATION_INFO)
            xbmc.executebuiltin('Container.Refresh')
    
    elif action == 'mark_unwatched':
        # Mark item as unwatched
        media_type = params.get('media_type', 'movie')
        media_id = params.get('media_id', '')
        season = params.get('season', '')
        episode = params.get('episode', '')
        
        if media_id:
            if media_type == 'movie':
                watch_history.mark_unwatched('movie', int(media_id))
                xbmcgui.Dialog().notification(ADDON_NAME, 'Odznačeno', xbmcgui.NOTIFICATION_INFO)
            elif season and episode:
                watch_history.mark_unwatched('tv', int(media_id), int(season), int(episode))
                xbmcgui.Dialog().notification(ADDON_NAME, 'Epizoda odznačena', xbmcgui.NOTIFICATION_INFO)
            xbmc.executebuiltin('Container.Refresh')


def show_recent_history(media_type):
    """Show recently watched items"""
    xbmcplugin.setPluginCategory(HANDLE, 'Naposledy sledovane')
    xbmcplugin.setContent(HANDLE, 'movies' if media_type == 'movie' else 'tvshows')
    
    history = watch_history.get_history(media_type, limit=20)
    
    if not history:
        # Show empty message but still end directory properly
        list_item = xbmcgui.ListItem(label='[COLOR gray]Zadna historie sledovani[/COLOR]')
        list_item.setProperty('IsPlayable', 'false')
        xbmcplugin.addDirectoryItem(HANDLE, '', list_item, isFolder=False)
        xbmcplugin.endOfDirectory(HANDLE)
        return
    
    for item in history:
        media_id = item.get('media_id')
        title = item.get('title', 'Unknown')
        year = item.get('year', '')
        poster = item.get('poster', '')
        
        # Build label
        label = title
        if year:
            label += f" ({year})"
        
        # Check if watched
        is_watched = watch_history.is_watched(media_type, media_id)
        if is_watched:
            label = f"[COLOR gray][OK][/COLOR] {label}"
        
        list_item = xbmcgui.ListItem(label=label)
        
        if poster:
            poster_url = f"https://image.tmdb.org/t/p/w500{poster}"
            list_item.setArt({'poster': poster_url, 'thumb': poster_url})
        
        info_tag = list_item.getVideoInfoTag()
        info_tag.setTitle(label)
        info_tag.setMediaType('movie' if media_type == 'movie' else 'tvshow')
        
        # Context menu for watched status
        context_menu = []
        if is_watched:
            context_menu.append(('Oznacit jako nezhlednute', 
                f'RunPlugin({get_url(action="mark_unwatched", media_type=media_type, media_id=str(media_id))})'))
        else:
            context_menu.append(('Oznacit jako zhlednute', 
                f'RunPlugin({get_url(action="mark_watched", media_type=media_type, media_id=str(media_id))})'))
        list_item.addContextMenuItems(context_menu)
        
        url = get_url(
            action='show_details',
            media_type=media_type,
            media_id=str(media_id),
            title=title
        )
        
        xbmcplugin.addDirectoryItem(HANDLE, url, list_item, isFolder=True)
    
    xbmcplugin.endOfDirectory(HANDLE)


def show_continue_watching():
    """Show items with playback progress"""
    xbmcplugin.setPluginCategory(HANDLE, 'Pokracovat v sledovani')
    xbmcplugin.setContent(HANDLE, 'videos')
    
    # Get items with progress from watch_history
    progress_items = watch_history.progress
    
    if not progress_items:
        # Show empty message but still end directory properly
        list_item = xbmcgui.ListItem(label='[COLOR gray]Zadne rozpracovane polozky[/COLOR]')
        list_item.setProperty('IsPlayable', 'false')
        xbmcplugin.addDirectoryItem(HANDLE, '', list_item, isFolder=False)
        xbmcplugin.endOfDirectory(HANDLE)
        return
    
    for key, progress_data in progress_items.items():
        parts = key.split(':')
        if len(parts) < 2:
            continue
        
        media_type = parts[0]
        media_id = parts[1]
        
        percent = progress_data.get('percent', 0)
        
        # Get title from TMDB
        if media_type == 'movie':
            details = tmdb.get_movie_details(media_id)
            title = details.get('title', 'Unknown') if details else 'Unknown'
            label = f"{title} - {percent:.0f}%"
        else:
            season = parts[2] if len(parts) > 2 else 1
            episode = parts[3] if len(parts) > 3 else 1
            details = tmdb.get_tvshow_details(media_id)
            title = details.get('name', 'Unknown') if details else 'Unknown'
            label = f"{title} S{season}E{episode} - {percent:.0f}%"
        
        list_item = xbmcgui.ListItem(label=label)
        list_item.setProperty('IsPlayable', 'true')
        
        if details and details.get('poster_path'):
            poster_url = f"https://image.tmdb.org/t/p/w500{details['poster_path']}"
            list_item.setArt({'poster': poster_url, 'thumb': poster_url})
        
        # URL to show sources
        if media_type == 'movie':
            url = get_url(
                action='show_details',
                media_type='movie',
                media_id=str(media_id),
                title=title
            )
        else:
            url = get_url(
                action='show_episode_sources',
                media_type='episode',
                media_id=str(media_id),
                title=title,
                season=str(season),
                episode=str(episode)
            )
        
        xbmcplugin.addDirectoryItem(HANDLE, url, list_item, isFolder=True)
    
    xbmcplugin.endOfDirectory(HANDLE)


if __name__ == '__main__':
    router(sys.argv[2][1:])
